//////////
//
//	File:		VRMovies.c
//
//	Contains:	Support for QuickTime movie playback in VR nodes.
//
//	Written by:	Tim Monroe
//				Some code borrowed from QTVRSamplePlayer by Bryce Wolfson.
//
//	Copyright:	 1996-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <16>		10/29/97	rtm		modified VRMoov_SetVideoGraphicsMode to use GetMovieIndTrackType
//	   <15>	 	09/18/97	rtm		added parameter to VRMoov_CheckForCompletedMovies
//	   <14>	 	06/24/97	rtm		modified VRMoov_DoIdle to service QuickTime movies
//									with video tracks in non-frontmost windows
//	   <13>	 	06/19/97	rtm		added support for scene-wide sound-only movies
//	   <12>	 	05/13/97	rtm		added VRMoov_PlayTransitionMovie
//	   <11>	 	04/28/97	rtm		reworked to support sound-only movie files as well
//	   <10>	 	04/28/97	rtm		added VRMoov_PlayMovie; reworked passim to use linked list
//	   <9>	 	03/17/97	rtm		added VRMoov_SetMovieBalanceAndVolume
//	   <8>	 	03/12/97	rtm		copied file from VRMovies project and integrated with VRScript
//	   <7>	 	03/06/97	rtm		started to implement video masking
//	   <6>	 	03/05/97	rtm		added VRMoov_SetChromaColor; added fChromaColor to app data record
//	   <5>	 	03/04/97	rtm		fixed compositing problems at back buffer edges
//	   <4>	 	03/03/97	rtm		added VRMoov_SetVideoGraphicsMode to handle compositing
//									without using an offscreen GWorld
//	   <3>	 	02/27/97	rtm		further development: borrowed some ideas from QTVRSamplePlayer;
//									added VRMoov_SetEmbeddedMovieWidth etc.
//	   <2>	 	12/12/96	rtm		further development: borrowed some ideas from BoxMoov demo 
//	   <1>	 	12/11/96	rtm		first file 
//	   
// This code supports playing QuickTime movies in QTVR panoramas. For movies with video tracks,
// this code draws the QuickTime movie frames into the back buffer, either directly
// or indirectly (via an offscreen GWorld). Direct drawing gives the best performance,
// but indirect drawing is necessary for some visual effects. 
//
//////////

// TO DO:
// + finish implementing on-the-fly rotation
// + finish video masking by custom 'hide' hot spots (so video goes *behind* such hot spots)
// + fix: when compositing (either w or w/o offscreen buffer) and movie is not initially visible
//   and movie rect intersects bb rect edge:
//	 panning to movie shows some junk in movie rect when movie first visible.
// + fix: when compositing w/o offscreen buffer, there's a blanking at repeat of movie

#pragma once

// header files
#include <stdlib.h>
#include <QDOffscreen.h>

#include "QTUtilities.h"
#include "QTVRUtilities.h"
#include "VRMovies.h"
#include "VRScript.h"

const RGBColor		kWhiteColor = {0xffff, 0xffff, 0xffff};		// white


//////////
//
// VRMoov_PlayMovie
// Start a QuickTime movie playing or stop one from playing.
//
//////////

void VRMoov_PlayMovie (
				WindowObject theWindowObject,
				UInt32 theNodeID,
				UInt32 theEntryID, 
				float thePanAngle,
				float theTiltAngle,
				float theScale, 
				float theWidth, 
				UInt32 theKeyRed, 
				UInt32 theKeyGreen, 
				UInt32 theKeyBlue, 
				Boolean theUseBuffer,
				Boolean theUseCenter,
				Boolean theUseKey,
				Boolean theUseHide,
				Boolean theUseDir,
				Boolean theUseVol,
				float theVolAngle,
				UInt32 theMode,
				UInt32 theOptions,
				char *thePathName)
{
	ApplicationDataHdl		myAppData;
	Boolean					myNeedPlayMovie = false;
	Boolean					myNeedLoadMovie = false;
	VRScriptMoviePtr		myPointer = NULL;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
	
	// see if this movie is already in our list of playing movies				
	myPointer = (VRScriptMoviePtr)VRScript_GetObjectByEntryID(theWindowObject, kVREntry_QTMovie, theEntryID);
		
	if (myPointer == NULL) {
		// this movie isn't in our list yet, so we'll need to add it to the list
		myNeedLoadMovie = true;
		myNeedPlayMovie = true;
	} else {
		// this movie is already in our list; theOptions determines how we handle this request:
		
		switch (theOptions) {
		
			case kVRMedia_PlayNew:
				// start the movie playing (whether it's already playing or not);
				// multiple movies aren't supported in QTVR 2.0, so we just map to kVRMedia_Continue
				myNeedPlayMovie = true;
				myNeedLoadMovie = true;
				break;
				
			case kVRMedia_Restart:
				// stop the current movie and then restart it; use the existing movie data
				StopMovie(myPointer->fMovie);
				myNeedPlayMovie = true;
				myNeedLoadMovie = false;
				break;
				
			case kVRMedia_ToggleStop:
				// toggle the movie's current play/stop state
				myNeedLoadMovie = false;
				if (!IsMovieDone(myPointer->fMovie)) {
					// stop the movie and get rid of movie list entry
					StopMovie(myPointer->fMovie);
					VRScript_DelistEntry(theWindowObject, (VRScriptGenericPtr)myPointer);
					myNeedPlayMovie = false;
				} else {
					// start the movie; use the existing movie data
					myNeedPlayMovie = true;
				}
				break;
				
			case kVRMedia_TogglePause:
				// toggle the movie's current play/pause state
				myNeedLoadMovie = false;
				if (GetMovieRate(myPointer->fMovie) != (Fixed)0) {
					// stop the movie
					StopMovie(myPointer->fMovie);
					myNeedPlayMovie = false;
				} else {
					// restart the movie; use the existing movie data
					myNeedPlayMovie = false;
					StartMovie(myPointer->fMovie);
				}
				break;
				
			case kVRMedia_Continue:
				// just let the movie already in progress continue
				myNeedPlayMovie = false;
				myNeedLoadMovie = false;
				break;
				
			case kVRMedia_Stop:
				// stop the current movie
				StopMovie(myPointer->fMovie);
				myNeedPlayMovie = false;
				myNeedLoadMovie = false;
				break;
				
			default:
				// unrecognized option
				// start the movie playing
				myNeedPlayMovie = true;
				myNeedLoadMovie = false;
				break;
		}
	}
	
	if (myNeedLoadMovie) {
		myPointer = VRScript_EnlistMovie (
						theWindowObject,
						theNodeID, 
						theEntryID,
						thePanAngle,
						theTiltAngle,
						theScale, 
						theWidth, 
						theKeyRed, 
						theKeyGreen, 
						theKeyBlue, 
						theUseBuffer,
						theUseCenter,
						theUseKey,
						theUseHide,
						theUseDir,
						theUseVol,
						theVolAngle,
						theMode,
						theOptions,
						thePathName);
	}

	if (myNeedPlayMovie) {
		if (myPointer != NULL) {
			FSSpec		myFSSpec;

			FSMakeFSSpec(0, 0L, c2pstr(thePathName), &myFSSpec);
			VRMoov_LoadEmbeddedMovie(&myFSSpec, theWindowObject, myPointer);
		}	
	}
}


//////////
//
// VRMoov_PlayTransitionMovie
// Play a QuickTime movie as a transition from one node to another.
//
//////////

void VRMoov_PlayTransitionMovie (WindowObject theWindowObject, UInt32 theOptions, char *thePathName)
{
	FSSpec		myFSSpec;
	short		myMovieFileRef;
	Movie		myMovie;
	Rect		myRect;
	GWorldPtr	myGWorld;
	GDHandle	myGDevice;
	OSErr		myErr = noErr;
	
	if (theWindowObject == NULL)
		return;
		
	FSMakeFSSpec(0, 0L, c2pstr(thePathName), &myFSSpec);
	
	// open the movie
	myErr = OpenMovieFile(&myFSSpec, &myMovieFileRef, fsRdPerm);
	if (myErr != noErr)
		return;

	myErr = NewMovieFromFile(&myMovie, myMovieFileRef, NULL, (StringPtr)NULL, newMovieActive, NULL);
	if ((myErr != noErr) || (myMovie == NULL))
		return;

	if (myMovieFileRef != 0)
		CloseMovieFile(myMovieFileRef);
		
	// set the movie GWorld
	GetMovieGWorld((**theWindowObject).fMovie, &myGWorld, &myGDevice);
	SetMovieGWorld(myMovie, myGWorld, myGDevice);
			
	// set the movie box to the size of the QTVR window			
	GetMovieBox((**theWindowObject).fMovie, &myRect);	
	SetMovieBox(myMovie, &myRect);
	
	// start the movie
	GoToBeginningOfMovie(myMovie);
	StartMovie(myMovie);
	
	// draw the movie image and play its sound
	if (theOptions == kVRMovie_PlayAll) 
		while (!IsMovieDone(myMovie)) {
			MoviesTask(myMovie, 0);
			SystemTask();
		}
		
	if (theOptions == kVRMovie_PlayTilClick) 
		while (!IsMovieDone(myMovie) && !Button()) {
			// note that this button test above doesn't grab the event, leaving a click in the queue...
			MoviesTask(myMovie, 0);
			SystemTask();
		}

	StopMovie(myMovie);
}


//////////
//
// VRMoov_LoadEmbeddedMovie
// Load the QuickTime movie in the specified file.
// Returns a Boolean to indicate success (true) or failure (false).
//
//////////

Boolean VRMoov_LoadEmbeddedMovie (FSSpec *theMovieFile, WindowObject theWindowObject, VRScriptMoviePtr theEntry)
{
	short					myMovieFileRef;
	Movie					myMovie;
	QTVRInstance			myInstance = NULL;
	GWorldPtr				myGWorld = NULL;
	CGrafPtr				mySavedPort;
	GDHandle				mySavedGDevice;
	Rect					myRect;
	ApplicationDataHdl		myAppData;
	OSErr					myErr = noErr;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(false);
		
	if (theEntry == NULL)
		return(false);
		
	// open the movie
	myErr = OpenMovieFile(theMovieFile, &myMovieFileRef, fsRdPerm);
	if (myErr != noErr)
		return(false);

	myErr = NewMovieFromFile(&myMovie, myMovieFileRef, NULL, (StringPtr)NULL, newMovieActive, NULL);
	if ((myErr != noErr) || (myMovie == NULL))
		return(false);

	if (myMovieFileRef != 0)
		CloseMovieFile(myMovieFileRef);

	GetMovieBox(myMovie, &myRect);
	MacOffsetRect(&myRect, -myRect.left, -myRect.top);
	SetMovieBox(myMovie, &myRect);

	// keep track of the movie and movie rectangle
	theEntry->fMovie = myMovie;
	theEntry->fMovieRect = myRect;
	
	// determine what kinds of media are in this movie
	theEntry->fQTMovieHasSound = QTUtils_IsMediaTypeInMovie(myMovie, SoundMediaType) || QTUtils_IsMediaTypeInMovie(myMovie, MusicMediaType);
	theEntry->fQTMovieHasVideo = QTUtils_IsMediaTypeInMovie(myMovie, VideoMediaType);
	if (theEntry->fQTMovieHasSound)
		theEntry->fMediaHandler = QTUtils_GetSoundMediaHandler(myMovie);

	// get rid of any existing offscreen graphics world
	if (theEntry->fMovieGWorld != NULL) {
		UnlockPixels(theEntry->fMoviePixMap);
		DisposeGWorld(theEntry->fMovieGWorld);
	}

	if (theEntry->fQTMovieHasVideo) {
		// get current port and GDevice
		GetGWorld(&mySavedPort, &mySavedGDevice);
		
		// clear out any existing custom cover function and reset the video media graphics mode
		// (these may have been modified for direct-screen drawing)
		SetMovieCoverProcs(myMovie, NULL, NULL, 0);
		VRMoov_SetVideoGraphicsMode(myMovie, theEntry, false);
		
		// if necessary, create an offscreen graphics world;
		// this is where we'll image the movie before copying it into the back buffer
		// (which allows us to do special effects)
		if (theEntry->fUseMovieGWorld) {
			myErr = NewGWorld(&myGWorld, 0, &myRect, NULL, NULL, 0L);
			if (myGWorld != NULL) 
				SetMovieGWorld(myMovie, myGWorld, GetGWorldDevice(myGWorld));
			theEntry->fMovieGWorld = myGWorld;
			theEntry->fMoviePixMap = GetGWorldPixMap(myGWorld);
			LockPixels(theEntry->fMoviePixMap);
		} else {			
			GetMovieGWorld((**theWindowObject).fMovie, &myGWorld, NULL);
			// set the video media graphics mode to drop out the chroma key color in a movie;
			// we also need to install an uncover function that doesn't erase the uncovered region
			if (theEntry->fCompositeMovie) {
				SetMovieCoverProcs(myMovie, NewMovieRgnCoverProc(VRMoov_CoverProc), NULL, (long)theWindowObject);
				VRMoov_SetVideoGraphicsMode(myMovie, theEntry, true);
			}
		}
		
		// install a back-buffer imaging procedure
		myInstance = (**theWindowObject).fInstance;
		if (myInstance != NULL)
			VRScript_InstallBackBufferImagingProc(myInstance, theWindowObject);
	}
	
	// set initial balance and volume
	if (theEntry->fQTMovieHasSound) {
		VRMoov_SetOneBalanceAndVolume(myMovie, theEntry->fMediaHandler, QTVRGetPanAngle(myInstance), QTVRGetTiltAngle(myInstance), theEntry->fMovieCenter.x, theEntry->fVolAngle);
		(**myAppData).fSoundHasChanged = true;
	}
		
	if (theEntry->fMode == kVRPlay_Loop) {
		// start the movie playing in a loop
		if (theEntry->fQTMovieHasVideo)
			SetMovieGWorld(myMovie, myGWorld, GetGWorldDevice(myGWorld));
		VRMoov_LoopEmbeddedMovie(myMovie);
	} else if (theEntry->fMode == kVRPlay_Once) {
		// play the movie once thru
		GoToBeginningOfMovie(myMovie);
		StartMovie(myMovie);
	}
	
	return(true);
}
	
	
//////////
//
// VRMoov_LoopEmbeddedMovie
// Start the QuickTime movie playing in a loop.
//
//////////

void VRMoov_LoopEmbeddedMovie (Movie theMovie)
{
	TimeBase		myTimeBase;

	// throw the movie into loop mode
	myTimeBase = GetMovieTimeBase(theMovie);
	SetTimeBaseFlags(myTimeBase, GetTimeBaseFlags(myTimeBase) | loopTimeBase);

	// start playing the movie
	StartMovie(theMovie);
}
	

//////////
//
// VRMoov_DoIdle
// Do any movie-related processing that can or should occur at idle time.
// Returns true if the caller should call QTVRUpdate, false otherwise.
//
//////////

Boolean VRMoov_DoIdle (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	VRScriptMoviePtr		myPointer;
	Boolean					myNeedUpdate = false;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(myNeedUpdate);
	
	// walk the linked list and service any embedded sound-only movies
	myPointer = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
	while (myPointer != NULL) {
		if (myPointer->fQTMovieHasSound && !(myPointer->fQTMovieHasVideo)) {
			MoviesTask(myPointer->fMovie, 0L);
		}
		myPointer = myPointer->fNextEntry;
	}
	
	// now service the (single) embedded video movie, if there is one	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer != NULL) {
		// trip the back buffer procedure, which internally calls MoviesTask;
		// note that this is necessary *only* if the window containing the embedded movie
		// isn't the frontmost window
		if ((**theWindowObject).fWindow != GetFrontMovieWindow())
			myNeedUpdate = true;
	}
	
	return(myNeedUpdate);
}
		
				
//////////
//
// VRMoov_DumpNodeMovies
// Stop playing all movies enlisted for the current node.
//
//////////

void VRMoov_DumpNodeMovies (WindowObject theWindowObject)
{
	VRMoov_DumpSelectedMovies(theWindowObject, kVRSelect_Node);
}


//////////
//
// VRMoov_DumpSceneMovies
// Stop playing all movies enlisted for the current scene.
//
//////////

void VRMoov_DumpSceneMovies (WindowObject theWindowObject)
{
	VRMoov_DumpSelectedMovies(theWindowObject, kVRSelect_Scene);
}


//////////
//
// VRMoov_DumpSelectedMovies
// Stop any existing embedded movies from playing and then clean up.
//
//////////

void VRMoov_DumpSelectedMovies (WindowObject theWindowObject, UInt32 theOptions)
{
	ApplicationDataHdl		myAppData;
	VRScriptMoviePtr		myPointer;
	VRScriptMoviePtr		myNext;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	// walk the movie list and dump any movies movies associated with this node
	myPointer = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
	while (myPointer != NULL) {
		
		myNext = myPointer->fNextEntry;
		if (((myPointer->fNodeID != kVRAnyNode) && (theOptions == kVRSelect_Node)) ||
			((myPointer->fNodeID == kVRAnyNode) && (theOptions == kVRSelect_Scene)))
			VRScript_DelistEntry(theWindowObject, (VRScriptGenericPtr)myPointer);
		
		myPointer = myNext;
	}
	
	// clear the existing back buffer imaging proc
	VRScript_RemoveBackBufferImagingProc((**theWindowObject).fInstance, theWindowObject);

	// make sure the back buffer is clean
	QTVRRefreshBackBuffer((**theWindowObject).fInstance, 0);

	return;
}
	

//////////
//
// VRMoov_GetEmbeddedVideo
// Return the first embedded QuickTime movie that has a video track.
//
//////////

VRScriptMoviePtr VRMoov_GetEmbeddedVideo (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	VRScriptMoviePtr		myPointer = NULL;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(myPointer);
		
	// walk our linked list of movies to find the target movie
	myPointer = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
	while (myPointer != NULL) {
		if (myPointer->fQTMovieHasVideo)
			return(myPointer);
		myPointer = myPointer->fNextEntry;
	}
	
	return(NULL);
}


//////////
//
// VRMoov_GetEmbeddedMovieWidth
// Get the width of the embedded movie.
//
//////////

float VRMoov_GetEmbeddedMovieWidth (WindowObject theWindowObject)
{
	float					myWidth = 0.0;
	VRScriptMoviePtr		myPointer;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer != NULL)
		myWidth = myPointer->fMovieWidth;
			
	return(myWidth);
}


//////////
//
// VRMoov_SetEmbeddedMovieWidth
// Set the width of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieWidth (WindowObject theWindowObject, float theWidth)
{
	QTVRInstance			myInstance;
	VRScriptMoviePtr		myPointer;
	
	if (theWindowObject == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer != NULL)
		myPointer->fMovieWidth = theWidth;

	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRScript_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_GetEmbeddedMovieCenter
// Get the center of the embedded movie.
//
//////////

void VRMoov_GetEmbeddedMovieCenter (WindowObject theWindowObject, QTVRFloatPoint *theCenter)
{
	VRScriptMoviePtr		myPointer;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer == NULL) {
		theCenter->x = 0.0;
		theCenter->y = 0.0;
	} else {
		theCenter->x = myPointer->fMovieCenter.x;
		theCenter->y = myPointer->fMovieCenter.y;
	}		
}


//////////
//
// VRMoov_SetEmbeddedMovieCenter
// Set the center of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieCenter (WindowObject theWindowObject, const QTVRFloatPoint *theCenter)
{
	QTVRInstance			myInstance;
	float					myX, myY;
	VRScriptMoviePtr		myPointer;

	if (theWindowObject == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	myX = theCenter->x;
	myY = theCenter->y;
	
	// subject the values passed in to the current view constraints
	QTVRWrapAndConstrain(myInstance, kQTVRPan, myX, &myX);
	QTVRWrapAndConstrain(myInstance, kQTVRTilt, myY, &myY);
			
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer != NULL) {
		myPointer->fMovieCenter.x = myX;
		myPointer->fMovieCenter.y = myY;
	}
	
	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRScript_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_GetEmbeddedMovieScale
// Get the scale of the embedded movie.
//
//////////

float VRMoov_GetEmbeddedMovieScale (WindowObject theWindowObject)
{
	float					myScale;
	VRScriptMoviePtr		myPointer;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer == NULL)
		myScale = 0.0;
	else
		myScale = myPointer->fMovieScale;

	return(myScale);
}


//////////
//
// VRMoov_SetEmbeddedMovieScale
// Set the scale factor of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieScale (WindowObject theWindowObject, float theScale)
{
	QTVRInstance			myInstance;
	VRScriptMoviePtr		myPointer;

	if (theWindowObject == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer != NULL)
		myPointer->fMovieScale = theScale;
	
	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRScript_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_GetEmbeddedMovieRect
// Get the rectangle of the embedded movie.
//
//////////

void VRMoov_GetEmbeddedMovieRect (WindowObject theWindowObject, Rect *theRect)
{
	VRScriptMoviePtr		myPointer;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer == NULL) {
		theRect->left = 0.0;
		theRect->top = 0.0;
		theRect->right = 0.0;
		theRect->bottom = 0.0;
	} else {
		theRect->left = myPointer->fMovieRect.left;
		theRect->top = myPointer->fMovieRect.top;
		theRect->right = myPointer->fMovieRect.right;
		theRect->bottom = myPointer->fMovieRect.bottom;
	}		
}


//////////
//
// VRMoov_SetEmbeddedMovieRect
// Set the rectangle of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieRect (WindowObject theWindowObject, const Rect *theRect)
{
	QTVRInstance			myInstance;
	VRScriptMoviePtr		myPointer;

	if (theWindowObject == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);
	if (myPointer != NULL) {
		myPointer->fMovieRect.right = theRect->right;
		myPointer->fMovieRect.bottom = theRect->bottom;
	}

	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRScript_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_SetAllBalanceAndVolume
// Set the balance and volume attenuation of all embedded QuickTime movies making sound.
//
//////////

void VRMoov_SetAllBalanceAndVolume (WindowObject theWindowObject, float thePan, float theTilt)
{
	ApplicationDataHdl		myAppData;
	VRScriptMoviePtr		myPointer;
	VRScript3DObjPtr		my3DObjPtr;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	// walk our linked list of embedded QuickTime movies and set the balance and volume of any localized sounds
	myPointer = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
	while (myPointer != NULL) {
		if (myPointer->fQTMovieHasSound)
			if (myPointer->fSoundIsLocalized)
				VRMoov_SetOneBalanceAndVolume(myPointer->fMovie, myPointer->fMediaHandler, thePan, theTilt, myPointer->fMovieCenter.x, myPointer->fVolAngle);
		myPointer = myPointer->fNextEntry;
	}
	
	// walk our linked list of 3D objects and set the balance and volume of any texture-mapped movies
	my3DObjPtr = (VRScript3DObjPtr)(**myAppData).fListArray[kVREntry_QD3DObject];
	while (my3DObjPtr != NULL) {
		if (my3DObjPtr->fTexture != NULL)
			if (my3DObjPtr->fTextureIsMovie)
				if ((**my3DObjPtr->fTexture).fMediaHandler != NULL)
					VRMoov_SetOneBalanceAndVolume((**my3DObjPtr->fTexture).fMovie, (**my3DObjPtr->fTexture).fMediaHandler, thePan, theTilt, QTVRUtils_Point3DToPanAngle(-1.0 * my3DObjPtr->fGroupCenter.x, my3DObjPtr->fGroupCenter.y, -1.0 * my3DObjPtr->fGroupCenter.z), QTVRUtils_DegreesToRadians(kVR_TextureMovieVolAngle));
		my3DObjPtr = my3DObjPtr->fNextEntry;
	}
	
}
	
	
//////////
//
// VRMoov_SetOneBalanceAndVolume
// Set the balance and volume attenuation of an embedded QuickTime movie.
//
//////////

void VRMoov_SetOneBalanceAndVolume (Movie theMovie, MediaHandler theMediaHandler, float thePan, float theTilt, float theMoviePan, float theVolAngle)
{
#pragma unused(theTilt)

	short			myValue;
	float			myPanDelta;
	float			myCosDelta;			// cosine of pan angle delta from movie center
	float			myCosLimit;			// cosine of attenuation cone limit angle
	
	myPanDelta = thePan - theMoviePan;

	// ***set the balance
	myValue = kQTMaxSoundVolume * sin(myPanDelta);
	MediaSetSoundBalance(theMediaHandler, myValue);
	
	// ***set the volume
	myCosDelta = cos(myPanDelta);
	myCosLimit = cos(theVolAngle);

	if (myCosDelta >= myCosLimit)
		// inside cone of attenuation, volume scales from 1.0 (at center) to 0.0 (at cone edge)
		myValue = kQTMaxSoundVolume * ((myCosDelta - myCosLimit) / (1 - myCosLimit));
	else
		// outside cone of attenuation, volume is 0.0;
		myValue = kNoVolume;
	
	if (myValue != GetMovieVolume(theMovie))
		SetMovieVolume(theMovie, myValue);
}


//////////
//
// VRMoov_BackBufferImagingProc
// The back buffer imaging procedure:
//	* get a frame of movie and image it into the back buffer
//	* also, do any additional compositing that might be desired
// This function is called by VRScript_BackBufferImagingProc.
//
//////////

PASCAL_RTN OSErr VRMoov_BackBufferImagingProc (QTVRInstance theInstance, Rect *theRect, UInt16 theAreaIndex, UInt32 theFlagsIn, UInt32 *theFlagsOut, WindowObject theWindowObject)
{
#pragma unused(theAreaIndex)

	ApplicationDataHdl		myAppData;
	Movie					myMovie;
	Boolean					myIsDrawing = theFlagsIn & kQTVRBackBufferRectVisible;
	VRScriptMoviePtr		myPointer;
	
	// assume we're not going to draw anything
	*theFlagsOut = 0;
	
	if ((theInstance == NULL) || (theWindowObject == NULL)) 
		return(paramErr);

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) 
		return(paramErr);

	myPointer = VRMoov_GetEmbeddedVideo(theWindowObject);

	// make sure we've got an active movie to play
	if (myPointer == NULL)
		return(paramErr);
		
	myMovie = myPointer->fMovie;
	if ((myMovie != NULL) && (!IsMovieDone(myMovie))) {
		// we have an embedded movie, so play it
		
		GWorldPtr			myGWorld, myMovGWorld;
		GDHandle			myGDevice, myMovGDevice;
		Rect				myRect;
		
		// get the current graphics world
		// (on entry, the current graphics world is [usually] set to the back buffer)
		GetGWorld(&myGWorld, &myGDevice);

		// get the embedded movie's graphics world
		GetMovieGWorld(myMovie, &myMovGWorld, &myMovGDevice);

		// make sure that the movie GWorld is set correctly
		if (myPointer->fUseMovieGWorld) {
			// we're using an offscreen graphics world,
			// so set movie's GWorld to be that offscreen graphics world
			if (myMovGWorld != myPointer->fMovieGWorld)
				SetMovieGWorld(myMovie, myPointer->fMovieGWorld, GetGWorldDevice(myPointer->fMovieGWorld));			
		} else {
			// we're not using an offscreen graphics world,
			// so set movie GWorld to be the back buffer;
			// (note that we call SetMovieGWorld only if we have to, for performance reasons)
			if ((myMovGWorld != myGWorld) || (myMovGDevice != myGDevice)) 
				SetMovieGWorld(myMovie, myGWorld, myGDevice);
				
			if (myIsDrawing) {
				// make sure the movie's rect matches our area of interest
				GetMovieBox(myMovie, &myRect);
				if (!MacEqualRect(&myRect, theRect))
					SetMovieBox(myMovie, theRect);
			} else {
				// if we're not visible, make sure we're not wasting time trying to draw
				MacSetRect(&myRect, 0, 0, 0, 0);
				SetMovieBox(myMovie, &myRect);
			}
				
		}
		
		// give the movie some time to play;
		// that is, draw a new frame into the movie's graphics world (and play movie sound)
		MoviesTask(myMovie, 0);
		
		// ***insertion (A) here***
		
		if (myIsDrawing) {
		
			// if we're visible, say we're drawing
			*theFlagsOut = kQTVRBackBufferFlagDidDraw;
			
			// now, if we're using an offscreen graphics world, copy it into the back buffer
			if (myPointer->fUseMovieGWorld) {
				PixMapHandle	myPixMap;
				
				myPixMap = GetGWorldPixMap(myGWorld);
				LockPixels(myPixMap);
				
				// set the chroma key color, if necessary
				if (myPointer->fCompositeMovie) {
					RGBBackColor(&(myPointer->fChromaColor));
					OpColor(&myPointer->fChromaColor);
				}
				
				// copy the current movie frame to the current graphics world
				CopyBits(	(BitMap *)(*myPointer->fMoviePixMap),
							(BitMap *)(*myPixMap),
							&(*myPointer->fMovieGWorld).portRect, 
							theRect,
							srcCopy | transparent, 
							NULL);

				// reset the chroma key color;
				// we need to do this because the buffer we just drew into might NOT actually
				// be the real back buffer (see Virtual Reality Programming With QuickTime VR, p. 1-154);
				// the copy between the intermediate buffer and the back buffer respects the current back color.
				if (myPointer->fCompositeMovie)
					RGBBackColor(&kWhiteColor);
					
				UnlockPixels(myPixMap);
			}
		}
		
	} else {
		// we don't have an embedded movie, so remove this back buffer imaging procedure
		VRScript_RemoveBackBufferImagingProc((**theWindowObject).fInstance, theWindowObject);
	}
	
	return(noErr);
}


//////////
//
// VRMoov_CoverProc
// The cover function of the embedded movie.
//
//////////

PASCAL_RTN OSErr VRMoov_CoverProc (Movie theMovie, RgnHandle theRegion, WindowObject theWindowObject)
{
#pragma unused(theMovie, theRegion, theWindowObject)

	return(noErr);
}


//////////
//
// VRMoov_SetVideoGraphicsMode
// Set the video media graphics mode of the embedded movie.
//
//////////

void VRMoov_SetVideoGraphicsMode (Movie theMovie, VRScriptMoviePtr theEntry, Boolean theSetVGM)
{
	Track		myTrack;
	Media		myMedia;

	if (theEntry == NULL)
		return;
		
	myTrack = GetMovieIndTrackType(theMovie, 1, VideoMediaType, movieTrackMediaType | movieTrackEnabledOnly);
	if (myTrack != NULL) {
		myMedia = GetTrackMedia(myTrack);
		if (theSetVGM)
			MediaSetGraphicsMode(GetMediaHandler(myMedia), srcCopy | transparent, &theEntry->fChromaColor);
		else
			MediaSetGraphicsMode(GetMediaHandler(myMedia), srcCopy, &kWhiteColor);
	} 
		
	return;
}


//////////
//
// VRMoov_GetFinishedMovie
// Get the first enlisted movie that is done playing.
//
//////////

VRScriptMoviePtr VRMoov_GetFinishedMovie (WindowObject theWindowObject)
{
	ApplicationDataHdl	myAppData;
	VRScriptMoviePtr	myPointer = NULL;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);	
	if (myAppData == NULL)
		return(NULL);
	
	// walk our linked list of movies to find the target movie
	myPointer = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
	while (myPointer != NULL) {
		if (IsMovieDone(myPointer->fMovie))
			return(myPointer);
		
		myPointer = myPointer->fNextEntry;
	}
	
	return(NULL);
}


//////////
//
// VRMoov_CheckForCompletedMovies
// Clean up any movies that are finished playing.
//
//////////

void VRMoov_CheckForCompletedMovies (WindowObject theWindowObject)
{
	VRScriptMoviePtr		myPointer = NULL;
	
	// delist any completed movies for the specified movie window
	if (theWindowObject != NULL)
		while ((myPointer = VRMoov_GetFinishedMovie(theWindowObject)) != NULL)
			VRScript_DelistEntry(theWindowObject, (VRScriptGenericPtr)myPointer);
}


//////////
//
// VRMoov_DumpEntryMem
// Release any memory associated with the specified list entry.
//
//////////

void VRMoov_DumpEntryMem (VRScriptMoviePtr theEntry)
{
	DisposeMovie(theEntry->fMovie);
	
	if (theEntry->fQTMovieHasVideo) {
		if (theEntry->fMoviePixMap != NULL)
			UnlockPixels(theEntry->fMoviePixMap);
		if (theEntry->fMovieGWorld != NULL)
			DisposeGWorld(theEntry->fMovieGWorld);
		if (theEntry->fHideRegion != NULL)
			DisposeRgn(theEntry->fHideRegion);
	}
	
	free(theEntry->fPathname);
}